function [dcdt, v, v_rel] = protmdl_n(t,c,f,D,tcycle,tfeed)

% General variables
AXP =   8.28;                % mol/m^-3           (total ATP + ADP)
ATP =   c(5);               % ATP: renamed for ease of use in equations
ADP = AXP - ATP;            % ADP: dependent on ATP concentration
NADX =  1.46;               % mol/m^-3           (total NAD + NADH)
NADH =  c(6);               % NADH: renamed for ease of use in equations
NAD = NADX - NADH;          % NAD: dependent on NADH concentration
rho_x = 26656;             % mol x per m^3      (density of biomass)
c_in =  [41.67;0;0;0];      % mol/m^-3           (inlet concentrations)


% Make distinction between extracellular and intracellular compounds
EC = [   1;     % Glucose
         1;     % Biomass 
         1;     % EtOH 
         1;     % Glycerol
         0;     % ATP
         0;     % NADH
         0;     % Pi
         0;     % G6P
         0;     % FBP
         0;     % Pyr
         0];    % Trehalose

EC_index = find(EC==1); IC_index = find(EC==0); 

% Implement Feast-famine operation (for chemostat: cycle_length = feast_length, for batch: D = 0 for the entire cycle)
D = D * tcycle/tfeed;

current_phase = t/tcycle - floor(t/tcycle);  

if current_phase > tfeed/tcycle
   D = 0;                % 1/h
end

% Kinetic model
% Enzyme activities, estimated with flux distribution of Heyland (2009) and
% the proteome allocation of De Godoy (2008)
k_c = [567.03;     % Upt
       223.07;     % UGlc
       127.84;     % LGlc
       229.1;      % Ferm
       30.721;     % ESnk
       3.5011;     % Resp
       5.9194;     % TrSn
       7.8115;     % TrDg
       0.57754;    % Grwt
       0.0155];    % Mtn
     
% Half-saturation constants (Km) [mol/m^3 or mM] (NaN = unused table position)
k_m = [5.1430       1.3863      NaN     NaN;    % Uptake     (Glc	ATP)
       6.5610       0.5180      5.5638  NaN;    % Uglc       (G6P	ATP     ATP_inhibition)
       2.4375       0.8942      15.8304 NaN;    % Lglc       (FBP	ADP     Pi)
       2.8680       0.01*NADX	NaN     NaN;    % Ferm       (PYR    NADH)
       0.5358       0.1166      NaN     NaN;    % ESnk       (FBP	NADH)
       2.6494       0.0994      0.1656  NaN;    % Mito       (PYR	NADH	Pi)
       3.2224       0.8445      16.5590 1.3863; % Treh       (G6P	ATP     Treh	ATP)
       0.75*AXP     0.0033      0.0033  0.0033; % �          (ATP	G6P     FBP     PYR)
       0.0150*AXP   NaN         NaN     NaN];   % mtn        (ATP)

% Hill exponents   
n = [2;         % Uglc vs ATP inhibition
     1.9;       % Ferm vs PYR
     25;        % Ferm vs NADH
     25;        % Resp vs ADP
     25;        % Resp vs Pi
     1.58;      % Trehalose synthesis vs G6P
     25;        % Growth vs ATP
     10;        % Growth vs G6P
     10;        % Growth vs FBP
     10;        % Growth vs PYR
     25];       % Growth vs NADH

   
% Stoichiometric matrix of reaction network
       %Upt  UGlc LGlc Ferm Esnk Mit1  Mit2 TrSn TrDg Grwt    Mtn 
S = [  -1    0    0    0    0    0      0    0    0    0       0;   % Glucose
        0    0    0    0    0    0      0    0    0    1       0;   % Biomass 
        0    0    0    1    0    0      0    0    0    0       0;   % Ethanol 
        0    0    0    0    1    0      0    0    0    0       0;   % Glycerol
       -1   -1    2    0    0    7/5    1.2 -1   -2   -2.0531 -1;   % ATP
        0    0    1   -1   -1    0     -1    0    0    0.2916  0;   % NADH
        0    0   -1    0    1   -7/5   -1.2  3    0    2.1930  1;   % Pi (growth is balanced such that all Pi is released)
        1   -1    0    0    0    0      0   -2    2   -0.0927  0;   % G6P
        0    1   -0.5  0   -0.5  0      0    0    0   -0.0236  0;   % FBP
        0    0    1   -1    0   -1/5    0    0    0   -0.1992  0;   % Pyruvate
        0    0    0    0    0    0      0    1   -1    0       0];  % Trehalose


% Uptake constraint:   ignore all uptake proteome allocation above a limit  
f_upt_max = 0.016;
f(1) = min(f(1),f_upt_max); 

% Respiration constraint
f_mit_max = 0.12;
f(6) = min(f(6),f_mit_max); 

% Calculate fluxes (mol/mol x/h)
v =    [f(1)*k_c(1)*c(1)/(k_m(1,1)+c(1))*ATP/(k_m(1,2)+ATP);                                                % Uptake
        f(2)*k_c(2)*c(8)/(c(8)+k_m(2,1))*ATP/(k_m(2,2)+ATP)*k_m(2,3)^n(1)/(k_m(2,3)^n(1)+ATP^n(1));         % Upper Glycolysis
        f(3)*k_c(3)*c(9)/(c(9)+k_m(3,1))*ADP/(ADP+k_m(3,2))*c(7)/(c(7)+k_m(3,3))*NAD/NADX;                  % Lower Glycolysis
        f(4)*k_c(4)*c(10)^n(2)/(k_m(4,1)^n(2)+c(10)^n(2));                                                  % Ethanol Fermentation
        f(5)*k_c(5)*c(9)/(k_m(5,1)+c(9))*NADH/(k_m(5,2)+NADH);                                              % Glycerol Electron Sink
        f(6)*k_c(6)*c(10)/(k_m(6,1)+c(10));                                                               % Respiration TCA
        f(6)*k_c(6)*NADH/(k_m(6,2)+NADH);                                                                   % Respiration NDE
        f(7)*k_c(7)*c(8)^n(6)/(c(8)^n(6)+k_m(7,1)^n(6)).*ATP./(ATP+k_m(7,2));                               % Trehalose Synthesis
        f(7)*k_c(8)*c(11)/(c(11)+k_m(7,3)).*ATP./(ATP+k_m(7,4));                                            % Trehalose Degradation
        f(8)*k_c(9)*ATP.^n(7)/(k_m(8,1).^n(7)+ATP.^n(7));                                                   % Growth
        k_c(10)*ATP/(k_m(9,1)+ATP)];                                                                        % Maintenance 
    
% "Safety valves"
v(4) = v(4) .* NADH^n(3) ./((k_m(4,2))^n(3) + NADH^n(3));       % Stop Fermentation when NADH is low
v(6) = v(6) .* ADP^n(4) ./((0.01*AXP)^n(4) + ADP^n(4));         % Stop Respiration when ADP < 0.01 AXP
v(7) = v(7) .* ADP^n(4) ./((0.01*AXP)^n(4) + ADP^n(4));         % Stop Respiration when ADP < 0.01 AXP
v(6) = v(6) .* c(7)^n(5) ./(k_m(6,3)^n(5) + c(7)^n(5));         % Stop Respiration when Pi is depleted
v(7) = v(7) .* c(7)^n(5) ./(k_m(6,3)^n(5) + c(7)^n(5));         % Stop Respiration when Pi is depleted
v(10) = v(10) .* c(8) .^n(8)/(k_m(8,2).^n(8)+c(8) .^n(8));      % Stop growth when G6P is depleted
v(10) = v(10) .* c(9) .^n(9)/(k_m(8,3).^n(9)+c(9) .^n(9));      % Stop growth when FBP is depleted
v(10) = v(10) .* c(10).^n(10)/(k_m(8,4).^n(10)+c(10).^n(10));   % Stop growth when PYR is depleted
v(10) = v(10) .* NAD.^n(11) ./((0.01*NADX).^n(11) + NAD.^n(11));% Stop growth when NAD < 0.05 NADX

% Correct mito2 such that the electron flow of mito1 and mito2 together does
% not exceed the total ETC capacity
ETCcap = f(6)*k_c(6);                                           % Calculate total capacity
ETCusage = v(6)+v(7);                                         % Calculate usage
if ETCusage > ETCcap                                            % If usage exceeds capacity, modify mito2 to match capacity
    v(7) = ETCcap-v(6);
end

% Switch off fermentation below the critical glucose concentration
if c(1) < 0.057
    v(4) = 0;
end

% Stress response penalty
f_mtn_req = 0.0203/(1+(c(1)/0.02)^21)+0.0618;
delta_f = f(9) - f_mtn_req;
if delta_f < 0 
    penalty = abs(delta_f)*100*0.0103;
    v(11) = v(11) + penalty;
end

% Calculate relative fluxes (with vmax = 1, diagnostic readout only)
v_max = [   f(1)*k_c(1);            % Uptake
            f(2)*k_c(2);            % Upper Glycolysis
            f(3)*k_c(3);            % Lower Glycolysis
            f(4)*k_c(4);            % Ethanol Fermentation
            f(5)*k_c(5);            % Glycerol Electron Sink
            f(6)*k_c(6);            % Respiration 1 (TCA cycle)
            f(6)*k_c(6)*6;          % Respiration 2 (external NADH respiration)
            f(7)*k_c(7);            % Trehalose synthesis
            f(7)*k_c(8);            % Trehalose degradation
            f(8)*k_c(9);            % Growth
            k_c(10)];               % Maintenance
        
v_rel = v./v_max; 
v_rel(7) = v(6)+v(7)./ETCcap; % Instead of resp2, represent the usage of the entire ETC


% Assemble system of ODEs
    
    % Extracellular balances: constant volume (washout)
    dcdt(EC_index) = (S(EC_index,:)*v)*c(2) + D*(c_in-c(EC_index));
    
    % Intracellular balances: volume scales with Cx
    dcdt(IC_index) = (S(IC_index,:)*v)*rho_x;

    dcdt = dcdt';
    
end
